/*********************************************************************
 *
 *	File Transfer Protocol (FTP) Client
 *  Module for Microchip TCP/IP Stack
 *	 -Provides ability to remotely upload new MPFS image (web pages) 
 *	  to external EEPROM or external Flash memory
 *	 -Reference: RFC 959
 *
 *********************************************************************
 * FileName:        FTP.c
 * Dependencies:    TCP, Tick, MPFS, FTPVerify() callback
 * Processor:       PIC18, PIC24F, PIC24H, dsPIC30F, dsPIC33F, PIC32
 * Compiler:        Microchip C32 v1.05 or higher
 *					Microchip C30 v3.12 or higher
 *					Microchip C18 v3.30 or higher
 *					HI-TECH PICC-18 PRO 9.63PL2 or higher
 * Company:         Microchip Technology, Inc.
 *
 * Software License Agreement
 *
 * Copyright (C) 2002-2009 Microchip Technology Inc.  All rights
 * reserved.
 *
 * Microchip licenses to you the right to use, modify, copy, and
 * distribute:
 * (i)  the Software when embedded on a Microchip microcontroller or
 *      digital signal controller product ("Device") which is
 *      integrated into Licensee's product; or
 * (ii) ONLY the Software driver source files ENC28J60.c, ENC28J60.h,
 *		ENCX24J600.c and ENCX24J600.h ported to a non-Microchip device
 *		used in conjunction with a Microchip ethernet controller for
 *		the sole purpose of interfacing with the ethernet controller.
 *
 * You should refer to the license agreement accompanying this
 * Software for additional information regarding your rights and
 * obligations.
 *
 * THE SOFTWARE AND DOCUMENTATION ARE PROVIDED "AS IS" WITHOUT
 * WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING WITHOUT
 * LIMITATION, ANY WARRANTY OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * MICROCHIP BE LIABLE FOR ANY INCIDENTAL, SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA, COST OF
 * PROCUREMENT OF SUBSTITUTE GOODS, TECHNOLOGY OR SERVICES, ANY CLAIMS
 * BY THIRD PARTIES (INCLUDING BUT NOT LIMITED TO ANY DEFENSE
 * THEREOF), ANY CLAIMS FOR INDEMNITY OR CONTRIBUTION, OR OTHER
 * SIMILAR COSTS, WHETHER ASSERTED ON THE BASIS OF CONTRACT, TORT
 * (INCLUDING NEGLIGENCE), BREACH OF WARRANTY, OR OTHERWISE.
 *
 *
 * Author               Date    Comment
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * Nilesh Rajbharti     4/23/01  Original        (Rev 1.0)
 * Nilesh Rajbharti     11/13/02 Fixed FTPServer()
 * Howard Schlunder		07/10/06 Added hash printing to FTP client
 * Howard Schlunder		07/20/06 Added FTP_RESP_DATA_NO_SOCKET error message
 * Elliott Wood			07/31/07 Minor updates for compatibility
  ********************************************************************/

/*
File modified from the original Microchip file by Mauro Grassi.

Added RETR command.

*/

#define __FTP_C

#include "h/TCPIPConfig.h"

#if defined(STACK_USE_FTP_SERVER)

#include "h/TCPIP.h"
#include "h/ff.h"
#include "h/common.h"
#include "h/helpers.h"
#include "h/maindemo.h"

#define MAX_FTP_ARGS                    (7u)
#define MAX_FTP_CMD_STRING_LEN          (255u)

typedef enum _SM_FTP
{
    SM_FTP_NOT_CONNECTED,
    SM_FTP_CONNECTED,
    SM_FTP_USER_NAME,
    SM_FTP_USER_PASS,
    SM_FTP_RESPOND
} SM_FTP;

typedef enum _SM_FTP_CMD
{
    SM_FTP_CMD_IDLE,
    SM_FTP_CMD_WAIT,
    SM_FTP_CMD_RECEIVE,
    SM_FTP_CMD_WAIT_FOR_DISCONNECT
} SM_FTP_CMD;

typedef enum _FTP_COMMAND
{
    FTP_CMD_USER,
    FTP_CMD_PASS,
    FTP_CMD_QUIT,
    FTP_CMD_STOR,
	FTP_CMD_RETR,	// added by M.G.
	FTP_CMD_NLST,	// added by M.G.
	FTP_CMD_LIST,	// added by M.G.
	FTP_CMD_DELE,	// added by M.G.
    FTP_CMD_PORT,
    FTP_CMD_ABORT,
    FTP_CMD_PWD,
	FTP_CMD_XPWD,	// added by M.G.
    FTP_CMD_CWD,
    FTP_CMD_TYPE,
    FTP_CMD_UNKNOWN,
    FTP_CMD_NONE,
} FTP_COMMAND;

// Each entry in following table must match with that of FTP_COMMAND enum.
static ROM char *FTPCommandString[] =
{
    "USER",                         // FTP_CMD_USER
    "PASS",                         // FTP_CMD_PASS
    "QUIT",                         // FTP_CMD_QUIT
    "STOR",                         // FTP_CMD_STOR
	"RETR",							// FTP_CMD_RETR			added by M.G.
	"NLST",							// FTP_CMD_NLST			added by M.G.
	"LIST",							// FTP_CMD_LIST			added by M.G.
	"DELE",							// FTP_CMD_DELE			added by M.G.
    "PORT",                         // FTP_CMD_PORT
    "ABOR",                         // FTP_CMD_ABORT
    "PWD ",                         // FTP_CMD_PWD
	"XPWD",							// FTP_CMD_PWD			added by M.G.
    "CWD ",                         // FTP_CMD_CWD
    "TYPE"                          // FTP_CMD_TYPE
};
#define FTP_COMMAND_TABLE_SIZE  ( sizeof(FTPCommandString)/sizeof(FTPCommandString[0]) )

typedef enum _FTP_RESPONSE
{
    FTP_RESP_BANNER,
    FTP_RESP_USER_OK,
    FTP_RESP_PASS_OK,
    FTP_RESP_QUIT_OK,
    FTP_RESP_STOR_OK,
    FTP_RESP_UNKNOWN,
    FTP_RESP_LOGIN,
    FTP_RESP_DATA_OPEN,
    FTP_RESP_DATA_READY,
    FTP_RESP_DATA_CLOSE,
	FTP_RESP_DATA_NO_SOCKET,
	FTP_RESP_PWD,
    FTP_RESP_OK,
	FTP_RESP_NO_FILE,					// added by M.G.
    FTP_RESP_NONE                       // This must always be the last
                                        // There is no corresponding string.
} FTP_RESPONSE;

extern void strcpyram2ram(BYTE*, BYTE*);
static BOOL DirectoryList(FTP_COMMAND);

// Each entry in following table must match with FTP_RESPONE enum
static ROM char *FTPResponseString[] =
{
    "220 Ready\r\n",                    // FTP_RESP_BANNER
    "331 Password required\r\n",        // FTP_RESP_USER_OK
    "230 Logged in\r\n",                // FTP_RESP_PASS_OK
    "221 Bye\r\n",                      // FTP_RESP_QUIT_OK
    "500 \r\n",                         // FTP_RESP_STOR_OK
    "502 Not implemented\r\n",          // FTP_RESP_UNKNOWN
    "530 Login required\r\n",           // FTP_RESP_LOGIN
    "150 Transferring data...\r\n",     // FTP_RESP_DATA_OPEN
    "125 Done\r\n",                    	// FTP_RESP_DATA_READY
    "226 Transfer Complete\r\n",        // FTP_RESP_DATA_CLOSE
	"425 Can't create data socket.\r\n",// FTP_RESP_DATA_NO_SOCKET
	"257 \"/\" is current\r\n",         // FTP_RESP_PWD
    "200 Ok\r\n",                       // FTP_RESP_OK
	"550 File Not Found.\r\n"			// FTP_RESP_NO_FILE			added by MG.
};

static union
{
    struct
    {
        unsigned char bUserSupplied : 1;
        unsigned char bLoggedIn: 1;
    } Bits;
    BYTE Val;
} FTPFlags;


static TCP_SOCKET       FTPSocket;      // Main ftp command socket.
static TCP_SOCKET       FTPDataSocket;  // ftp data socket.
static WORD_VAL         FTPDataPort;    // ftp data port number as supplied by client

static SM_FTP           smFTP;          // ftp server FSM state
static SM_FTP_CMD       smFTPCommand;   // ftp command FSM state

static FTP_COMMAND      FTPCommand;
static FTP_RESPONSE     FTPResponse;

static BYTE             FTPUser[FTP_USER_NAME_LEN];
static BYTE             FTPString[MAX_FTP_CMD_STRING_LEN+2];
static BYTE             FTPStringLen;
static BYTE             *FTP_argv[MAX_FTP_ARGS];    // Parameters for a ftp command
static BYTE             FTP_argc;       			// Total number of params for a ftp command
static TICK             lastActivity;   			// Timeout keeper.


//static MPFS             FTPFileHandle;
static	FIL				FTPFileHandle;
static 	DIR 			fdir;
static 	char			FTPCWD[MAX_FTP_CMD_STRING_LEN];
static  char 			responseBuffer[FILE_BUFFER_SIZE];
static  char			fullPathName[MAX_FTP_CMD_STRING_LEN*2];
static  FILINFO			fno;

// Private helper functions.
static void ParseFTPString(void);
static FTP_COMMAND ParseFTPCommand(BYTE *cmd);
static void ParseFTPString(void);
static BOOL ExecuteFTPCommand(FTP_COMMAND cmd);
static BOOL PutFile(void);
static BOOL GetFile(void);
static BOOL Quit(void);
static BOOL DeleteFile(void);
IP_ADDR lastFTPIP;
DWORD lastFTP;
long lastFTPResult;
DWORD SNTPGetSeconds(void);
void DisplayIPValue(char*, IP_ADDR);        
#define FTP_PUT_ENABLED
long FTPTimeout;

/*********************************************************************
 * Function:        void FTPInit(void)
 *
 * PreCondition:    TCP module is already initialized.
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        Initializes internal variables of FTP
 *
 * Note:
 ********************************************************************/
void FTPInit(void)
{

	FTPCWD[0]='/';
	FTPCWD[1]='\0';

	FTPSocket = INVALID_SOCKET;
    FTPSocket       = TCPOpen(0, TCP_OPEN_SERVER, (WORD)ftpPort, TCP_PURPOSE_FTP_COMMAND);

	// If this trap ever gets entered, it means your TCPIPConfig.h file 
	// needs to be edited to make a TCP_PURPOSE_FTP_COMMAND socket 
	// available, as defined by the TCPSocketInitializer[] array.  Make 
	// sure that a TCP_PURPOSE_FTP_DATA socket is also available to 
	// use the FTP Server.
    if(FTPSocket == INVALID_SOCKET)
    	while(1);

	FTPDataSocket	= INVALID_SOCKET;
    smFTP           = SM_FTP_NOT_CONNECTED;
    FTPStringLen    = 0;
    FTPFlags.Val    = 0;
    FTPDataPort.Val = (WORD)ftpDataPort;
}

/*********************************************************************
 * Function:        void FTPServer(void)
 *
 * PreCondition:    FTPInit() must already be called.
 *
 * Input:           None
 *
 * Output:          Opened FTP connections are served.
 *
 * Side Effects:    None
 *
 * Overview:
 *
 * Note:            This function acts as a task (similar to one in
 *                  RTOS).  This function performs its task in
 *                  co-operative manner.  Main application must call
 *                  this function repeatdly to ensure all open
 *                  or new connections are served on time.
 ********************************************************************/
BOOL FTPServer(void)
{
    BYTE v;
	int i;
    TICK currentTick;

    if ( !TCPIsConnected(FTPSocket) )
    {
        FTPStringLen    = 0;
        FTPCommand      = FTP_CMD_NONE;
        smFTP           = SM_FTP_NOT_CONNECTED;
        FTPFlags.Val    = 0;
        smFTPCommand    = SM_FTP_CMD_IDLE;
		if(FTPDataSocket != INVALID_SOCKET)
		{
			TCPDisconnect(FTPDataSocket);
			FTPDataSocket = INVALID_SOCKET;
		}
        //putrsUART("FT#A");
        return TRUE;
    }

    if ( TCPIsGetReady(FTPSocket) )
    {
        lastActivity    = TickGet();

        while( TCPGet(FTPSocket, &v ) )
        {
            FTPString[FTPStringLen++]   = v;
            if ( FTPStringLen == MAX_FTP_CMD_STRING_LEN )
                FTPStringLen            = 0;
        }


        if ( v == '\n' )
        {
            FTPString[FTPStringLen]     = '\0';
            FTPStringLen                = 0;
		    ParseFTPString();
            FTPCommand                  = ParseFTPCommand(FTP_argv[0]);
			//putrsUART("Command: ");
			//for(i=0; i<7; i++)
			//{
			//putrsUART(FTP_argv[i]);
			//putrsUART(",");
			//}
			//putrsUART("\r\n");
	    }
    }
    else if (smFTP!=SM_FTP_NOT_CONNECTED)
    {
        currentTick = TickGet();
        currentTick = TickGetDiff(currentTick, lastActivity);
        currentTick /= TICK_SECOND;
        if((currentTick>=FTPTimeout)&&(FTPTimeout>0))
        {
            lastActivity                = TickGet();
            FTPCommand                  = FTP_CMD_QUIT;
            smFTP                       = SM_FTP_CONNECTED;
        }
        //putrsUART("FT#B");
    }

    switch(smFTP)
    {
    case SM_FTP_NOT_CONNECTED:
        //putrsUART("FT#C");
        FTPResponse = FTP_RESP_BANNER;
        lastActivity = TickGet();
        // No break - Continue...

    case SM_FTP_RESPOND:
SM_FTP_RESPOND_Label:
        //putrsUART("FT#D");
		// Make sure there is enough TCP TX FIFO space to put our response
		if(FTPResponse != FTP_RESP_PWD)
		{
			memcpy((void*)responseBuffer, (ROM void*)FTPResponseString[FTPResponse], sizeof(responseBuffer));
		} else
		{
			strcpypgm2ram(responseBuffer, "257 \"");
			i=4;
			while(FTPCWD[i-4]!='\0')
			{
					responseBuffer[i]=FTPCWD[i-4];
					i++;
			}
			strcpypgm2ram(&responseBuffer[i], "\" is current\r\n");
		}
        if(TCPIsPutReady(FTPSocket) < strlen(responseBuffer))
			return TRUE;
		TCPPutArray(FTPSocket, (BYTE*)responseBuffer, strlen(responseBuffer));
		TCPFlush(FTPSocket);
		FTPResponse = FTP_RESP_NONE;
		smFTP = SM_FTP_CONNECTED;
        // No break - this will speed up little bit

    case SM_FTP_CONNECTED:
     
        if ( FTPCommand != FTP_CMD_NONE )
        {
            if ( ExecuteFTPCommand(FTPCommand) )
            {
                   
                if (FTPResponse != FTP_RESP_NONE )
                    {
                        //putrsUART("FT#F");
                        smFTP = SM_FTP_RESPOND;
                    }    
                else if ( FTPCommand == FTP_CMD_QUIT )
                    {
                        //putrsUART("FT#G");
                        smFTP = SM_FTP_NOT_CONNECTED;
                    
                    }

                //putrsUART("FT#0\r\n");
                FTPCommand = FTP_CMD_NONE;
                smFTPCommand = SM_FTP_CMD_IDLE;
            }
            else if ( FTPResponse != FTP_RESP_NONE )
            {
                smFTP = SM_FTP_RESPOND;
                //putrsUART("FT#1\r\n");
                goto SM_FTP_RESPOND_Label;
            }
        }
        break;

		default:
			break;
    }
    return TRUE;
}

static BOOL ExecuteFTPCommand(FTP_COMMAND cmd)
{
	int i, j;
    SOCKET_INFO* mySocketInfo;
    
    switch(cmd)
    {
    case FTP_CMD_USER:
        FTPFlags.Bits.bUserSupplied = TRUE;
        FTPFlags.Bits.bLoggedIn = FALSE;
        FTPResponse = FTP_RESP_USER_OK;
		FTPCWD[0]='/';
		FTPCWD[1]='\0';
        strncpy((char*)FTPUser, (char*)FTP_argv[1], sizeof(FTPUser));
        break;

    case FTP_CMD_PASS:
        if ( !FTPFlags.Bits.bUserSupplied )
            FTPResponse = FTP_RESP_LOGIN;
        else
        {
            lastFTP = SNTPGetSeconds();
            mySocketInfo = TCPGetRemoteInfo(FTPSocket);
            lastFTPIP = mySocketInfo->remote.IPAddr;
            logString("FTP: Access to FTP Server from IP: ", 1);
            DisplayIPValue(iBuffer, lastFTPIP);
            logString(iBuffer, 0);
            if ( FTPVerify(FTPUser, FTP_argv[1]) )
            {
                FTPFlags.Bits.bLoggedIn = TRUE;
                FTPResponse = FTP_RESP_PASS_OK;
				logString(" granted...\r\n", 0);
				lastFTPResult=1;
            }
            else
			{
                FTPResponse = FTP_RESP_LOGIN;
				logString(" declined...\r\n", 0);
				lastFTPResult=0;
			}
        }
        break;

    case FTP_CMD_QUIT:
        return Quit();

    case FTP_CMD_PORT:
        FTPDataPort.v[1] = (BYTE)myatol((char*)FTP_argv[5]);
        FTPDataPort.v[0] = (BYTE)myatol((char*)FTP_argv[6]);
        FTPResponse = FTP_RESP_OK;
        break;

    case FTP_CMD_STOR:
		 return PutFile();
	
	case FTP_CMD_RETR:
		 return GetFile();
        
    case FTP_CMD_PWD:
	case FTP_CMD_XPWD:
    	FTPResponse = FTP_RESP_PWD;
	   	break;

	case FTP_CMD_LIST:
	case FTP_CMD_NLST:
		return DirectoryList(cmd);

	case FTP_CMD_DELE:
		if(DeleteFile())FTPResponse = FTP_RESP_OK; else FTPResponse=FTP_RESP_NO_FILE;
		return TRUE;

    case FTP_CMD_CWD:
		if(!strcmppgm2ram((char*)FTP_argv[1], ".."))
		{
			i=strlen(FTPCWD)-2;
			while(i>0)
			{
			if(FTPCWD[i]=='/')break;
			i--;
			}
			if(i>0)FTPCWD[i]='\0'; else strcpypgm2ram(FTPCWD, "/");
		} 
		else
		if(!strcmppgm2ram((char*)FTP_argv[1], "/"))
		{
			strcpypgm2ram(FTPCWD, "/");
		}
		else
		{
		i=strlen(FTPCWD);
		j=strlen((char*)FTP_argv[1]);
	  	 if((i+j+1)<MAX_FTP_CMD_STRING_LEN)
		 {
			if(FTPCWD[i-1]!='/')FTPCWD[i++]='/';
			strcpyram2ram((BYTE*)&FTPCWD[i], (BYTE*)FTP_argv[1]);
		 }
		}
		//putrsUART("Here at CWD and ");
		//putrsUART(FTP_argv[1]);
		//putrsUART("\r\n");
		FTPResponse = FTP_RESP_OK;
		break;

    case FTP_CMD_TYPE:
       	FTPResponse = FTP_RESP_OK;
    	break;

    case FTP_CMD_ABORT:
        FTPResponse = FTP_RESP_OK;
        if ( FTPDataSocket != INVALID_SOCKET )
		{
			TCPDisconnect(FTPDataSocket);
			FTPDataSocket = INVALID_SOCKET;
		}
        break;

    default:
        FTPResponse = FTP_RESP_UNKNOWN;
        break;
    }
    return TRUE;
}

static BOOL Quit(void)
{
    switch(smFTPCommand)
    {
    case SM_FTP_CMD_IDLE:
#if defined(FTP_PUT_ENABLED)
        if ( smFTPCommand == SM_FTP_CMD_RECEIVE )
            //MPFSClose();
			f_close(&FTPFileHandle);
#endif

        if ( FTPDataSocket != INVALID_SOCKET )
        {
#if defined(FTP_PUT_ENABLED)
            //MPFSClose();
			f_close(&FTPFileHandle);
#endif
            TCPDisconnect(FTPDataSocket);
            smFTPCommand = SM_FTP_CMD_WAIT;
        }
        else
            goto Quit_Done;
        break;

    case SM_FTP_CMD_WAIT:
        if ( !TCPIsConnected(FTPDataSocket) )
        {
Quit_Done:
            FTPResponse = FTP_RESP_QUIT_OK;
            smFTPCommand = SM_FTP_CMD_WAIT_FOR_DISCONNECT;
			logString("FTP: FTP Server logout...\r\n", 1);
        }
        break;

    case SM_FTP_CMD_WAIT_FOR_DISCONNECT:
        if ( TCPIsPutReady(FTPSocket) )
        {
            if ( TCPIsConnected(FTPSocket) )
                TCPDisconnect(FTPSocket);
        }
        break;
	default:
		break;
    }
    return FALSE;
}

void getFullPathName(BYTE* fileNameString)
{
	int n;
	strcpyram2ram((BYTE*)fullPathName, (BYTE*)FTPCWD);
	n=strlen(fullPathName);
	if(n>0)
	{
	if(fullPathName[n-1]!='/'){ fullPathName[n++]='/'; }
	}
	strcpyram2ram((BYTE*)&fullPathName[n], (BYTE*)fileNameString);
}

static BOOL GetFile(void)
{
	UINT n;

    switch(smFTPCommand)
    {
    case SM_FTP_CMD_IDLE:
        if ( !FTPFlags.Bits.bLoggedIn )
        {
            FTPResponse     = FTP_RESP_LOGIN;
            return TRUE;
        }
        else
        {
			FTPResponse     = FTP_RESP_DATA_OPEN;
            FTPDataSocket   = TCPOpen((PTR_BASE)&TCPGetRemoteInfo(FTPSocket)->remote, TCP_OPEN_NODE_INFO, FTPDataPort.Val, TCP_PURPOSE_FTP_DATA);
			// Make sure that a valid socket was available and returned
			// If not, return with an error
			if(FTPDataSocket == INVALID_SOCKET)
			{
	            FTPResponse = FTP_RESP_DATA_NO_SOCKET;
	            return TRUE;
			}
			smFTPCommand = SM_FTP_CMD_WAIT;
        }
        break;

    case SM_FTP_CMD_WAIT:
        if (TCPIsConnected(FTPDataSocket) )
        {
			//FTPFileHandle   = MPFSFormat();
			lastActivity    = TickGet();	
			getFullPathName(FTP_argv[1]);
			if((f_open(&FTPFileHandle, (const char*)fullPathName, FA_READ | FA_OPEN_EXISTING))==FR_OK)
			{
			smFTPCommand = SM_FTP_CMD_RECEIVE;
			} else
			{
			TCPDisconnect(FTPDataSocket);
            FTPDataSocket   = INVALID_SOCKET;
            FTPResponse     = FTP_RESP_NO_FILE;
			return TRUE;
			}
        }
        break;

    case SM_FTP_CMD_RECEIVE:
        if (TCPIsPutReady(FTPDataSocket)>=FILE_BUFFER_SIZE)
        {
            // Reload timeout timer.
            lastActivity    = TickGet();
            // MPFSPutBegin(FTPFileHandle);
            // MPFSPut(v);
			f_read(&FTPFileHandle, fileBuffer, FILE_BUFFER_SIZE, &n);
			TCPPutArray(FTPDataSocket, fileBuffer, n);
			if(n==FILE_BUFFER_SIZE)return FALSE;
            //FTPFileHandle = MPFSPutEnd();
			f_close(&FTPFileHandle);
            TCPDisconnect(FTPDataSocket);
            FTPDataSocket   = INVALID_SOCKET;
            FTPResponse     = FTP_RESP_DATA_CLOSE;
            return TRUE;
        }
        else if ( !TCPIsConnected(FTPDataSocket) )
        {
            // MPFSClose();
			f_close(&FTPFileHandle);
            TCPDisconnect(FTPDataSocket);
            FTPDataSocket   = INVALID_SOCKET;
            FTPResponse     = FTP_RESP_DATA_CLOSE;
            return TRUE;
        }
		default:
			break;

    }
    return FALSE;
}

static BOOL DirectoryList(FTP_COMMAND x)
{
	WORD n;
	UINT u;
	FRESULT res;
	static int nNames;
	char* lfn;
	
    switch(smFTPCommand)
    {
    case SM_FTP_CMD_IDLE:
        if (!FTPFlags.Bits.bLoggedIn)
        {
            FTPResponse     = FTP_RESP_LOGIN;
            return TRUE;
        }
        else
        {
			FTPResponse     = FTP_RESP_DATA_OPEN;
            FTPDataSocket   = TCPOpen((PTR_BASE)&TCPGetRemoteInfo(FTPSocket)->remote, TCP_OPEN_NODE_INFO, FTPDataPort.Val, TCP_PURPOSE_FTP_DATA);
			// Make sure that a valid socket was available and returned
			// If not, return with an error
			if(FTPDataSocket == INVALID_SOCKET)
			{
	            FTPResponse = FTP_RESP_DATA_NO_SOCKET;
	            return TRUE;
			}
			smFTPCommand = SM_FTP_CMD_WAIT;
        }
        break;

    case SM_FTP_CMD_WAIT:
        if ( TCPIsConnected(FTPDataSocket) )
        {
			//FTPFileHandle   = MPFSFormat();
			lastActivity    = TickGet();
			//putrsUART("Opening Directory: ");
			//putrsUART(FTPCWD);
			//putrsUART("\r\nResponse: ");
			res=f_opendir(&fdir, (char*)FTPCWD);
			//disA(res);
			//putrsUART("\r\n");
			if(res==FR_OK)
			{
			smFTPCommand    = SM_FTP_CMD_RECEIVE;
			nNames=0;
			} 
			else
			{
            TCPDisconnect(FTPDataSocket);
            FTPDataSocket   = INVALID_SOCKET;
            FTPResponse     = FTP_RESP_NO_FILE;
            return TRUE;
			}
        }
        break;

    case SM_FTP_CMD_RECEIVE:
		u=TCPIsPutReady(FTPDataSocket);
        if (u>32)
        {
            // Reload timeout timer.
            lastActivity = TickGet();
			res=f_readdir(&fdir, &fno);
	        if (res != FR_OK || fno.fname[0] == 0)
			{
				// end of listing...
				fileBuffer[0]='\r';
				fileBuffer[1]='\n';
				fileBuffer[2]='\0';
    			TCPPutArray(FTPDataSocket, (BYTE*)fileBuffer, 3);
    	        TCPDisconnect(FTPDataSocket);
	            FTPDataSocket   = INVALID_SOCKET;
        	    FTPResponse     = FTP_RESP_DATA_CLOSE;
    	       	 return TRUE;
			}
			nNames++;
			fileBuffer[0]='\t';
			if(((*fno.lfname)=='\0')||(x==FTP_CMD_NLST))lfn=fno.fname; else lfn=fno.lfname;
			if(fno.fattrib & AM_DIR)
			{
			fileBuffer[1]='[';
			strcpyram2ram(&fileBuffer[2], (BYTE*)lfn);
			n=strlen((char*)fileBuffer);
			fileBuffer[n++]=']';
			}
			else
			{
			 strcpyram2ram(&fileBuffer[1], (BYTE*)lfn);
			 n=strlen((char*)fileBuffer);
			}
			while(n<16)
			{
			fileBuffer[n++]=' ';
			}

  		    if(x==FTP_CMD_LIST)
			{
			strcpypgm2ram((char*)&fileBuffer[n], "\t\t\t");
			n=strlen((char*)fileBuffer);
			if(!(fno.fattrib & AM_DIR))
				{
				ultoa(fno.fsize, &fileBuffer[n]);
				n=strlen((char*)fileBuffer);
				}
			}

			if((nNames>=4)||(x==FTP_CMD_LIST))
			{
			fileBuffer[n++]='\r';
			fileBuffer[n++]='\n';
			nNames=0;
			}

			fileBuffer[n++]='\0';
			n=TCPPutArray(FTPDataSocket, (BYTE*)fileBuffer, n);
			return FALSE;
        }
        else if ( !TCPIsConnected(FTPDataSocket) )
        {
            //MPFSClose();
            TCPDisconnect(FTPDataSocket);
            FTPDataSocket   = INVALID_SOCKET;
            FTPResponse     = FTP_RESP_DATA_CLOSE;
            return TRUE;
        }
	default:
			break;
    }
    return FALSE;
}

static BOOL PutFile(void)
{
	WORD n;
    UINT u;
	FRESULT res;

    switch(smFTPCommand)
    {
    case SM_FTP_CMD_IDLE:
        if ( !FTPFlags.Bits.bLoggedIn )
        {
            FTPResponse     = FTP_RESP_LOGIN;
            return TRUE;
        }
        else
        {
			FTPResponse     = FTP_RESP_DATA_OPEN;
            FTPDataSocket   = TCPOpen((PTR_BASE)&TCPGetRemoteInfo(FTPSocket)->remote, TCP_OPEN_NODE_INFO, FTPDataPort.Val, TCP_PURPOSE_FTP_DATA);
			// Make sure that a valid socket was available and returned
			// If not, return with an error
			if(FTPDataSocket == INVALID_SOCKET)
			{
	            FTPResponse = FTP_RESP_DATA_NO_SOCKET;
	            return TRUE;
			}
			smFTPCommand = SM_FTP_CMD_WAIT;
        }
        break;

    case SM_FTP_CMD_WAIT:
        if ( TCPIsConnected(FTPDataSocket) )
        {
			//FTPFileHandle   = MPFSFormat();
			lastActivity    = TickGet();
			res=f_opendir(&fdir, FTPCWD);
			//putrsUART("Open dir: ");
			//putrsUART(FTPCWD);
			//disA(res);
			//putrsUART("\r\n");
			if(res==FR_NO_PATH)
			{
			res=f_mkdir(FTPCWD);			// make the directory in case it doesn't exist.
			//putrsUART("Make Directory ");
			//putrsUART(FTPCWD);
			//disA(res);
			//putrsUART("\r\n");
			}
			if(res==FR_OK)
			{
			getFullPathName(FTP_argv[1]);
			if((f_open(&FTPFileHandle, (const char*)fullPathName, FA_CREATE_ALWAYS | FA_WRITE))==FR_OK)
			{
			smFTPCommand    = SM_FTP_CMD_RECEIVE;
			} else
			{
            TCPDisconnect(FTPDataSocket);
            FTPDataSocket   = INVALID_SOCKET;
            FTPResponse     = FTP_RESP_NO_FILE;
            return TRUE;
			}
			}
			else
			{
 			TCPDisconnect(FTPDataSocket);
            FTPDataSocket   = INVALID_SOCKET;
            FTPResponse     = FTP_RESP_NO_FILE;
            return TRUE;
			}
        }
        break;

    case SM_FTP_CMD_RECEIVE:
		u=TCPIsGetReady(FTPDataSocket);
        if (u>0)
        {
            // Reload timeout timer.
            lastActivity    = TickGet();
            n=TCPGetArray(FTPDataSocket, fileBuffer, u);
			f_write(&FTPFileHandle, fileBuffer, n, &u);
			return FALSE;
        }
        else if ( !TCPIsConnected(FTPDataSocket) )
        {
            //MPFSClose();
			f_close(&FTPFileHandle);
            TCPDisconnect(FTPDataSocket);
            FTPDataSocket   = INVALID_SOCKET;
            FTPResponse     = FTP_RESP_DATA_CLOSE;
            return TRUE;
        }
	default:
			break;
    }
    return FALSE;
}


static BOOL DeleteFile(void)
{
	FRESULT res;
		
	getFullPathName(FTP_argv[1]);
	res=f_unlink(fullPathName);
	if(res==FR_OK)return TRUE; else return FALSE;
}

/*
static BOOL DeleteFile(void)
{
	FRESULT res;

    switch(smFTPCommand)
    {
    case SM_FTP_CMD_IDLE:
        if ( !FTPFlags.Bits.bLoggedIn )
        {
            FTPResponse     = FTP_RESP_LOGIN;
            return TRUE;
        }
        else
        {
			FTPResponse     = FTP_RESP_DATA_OPEN;
            FTPDataSocket   = TCPOpen((PTR_BASE)&TCPGetRemoteInfo(FTPSocket)->remote, TCP_OPEN_NODE_INFO, FTPDataPort.Val, TCP_PURPOSE_FTP_DATA);
			// Make sure that a valid socket was available and returned
			// If not, return with an error
			if(FTPDataSocket == INVALID_SOCKET)
			{
	            FTPResponse = FTP_RESP_DATA_NO_SOCKET;
	            return TRUE;
			}
			smFTPCommand = SM_FTP_CMD_WAIT;
        }
        break;

    case SM_FTP_CMD_WAIT:
        if ( TCPIsConnected(FTPDataSocket) )
        {
			//FTPFileHandle   = MPFSFormat();
			lastActivity    = TickGet();
			getFullPathName(FTP_argv[1]);
			res=f_unlink(fullPathName);
			putrsUART("Here now");
			if(res==FR_OK)
			{
			smFTPCommand    = SM_FTP_CMD_RECEIVE;
			} else
			{
            TCPDisconnect(FTPDataSocket);
            FTPDataSocket   = INVALID_SOCKET;
            FTPResponse     = FTP_RESP_NO_FILE;
            return TRUE;
			}
			}
			else
			{
 			TCPDisconnect(FTPDataSocket);
            FTPDataSocket   = INVALID_SOCKET;
            FTPResponse     = FTP_RESP_NO_FILE;
            return TRUE;
			}
        break;

    case SM_FTP_CMD_RECEIVE:
			TCPDisconnect(FTPDataSocket);
            FTPDataSocket   = INVALID_SOCKET;
            FTPResponse     = FTP_RESP_OK;
            return TRUE;
        
	default:
			break;
    }
    return FALSE;
}
*/

static FTP_COMMAND ParseFTPCommand(BYTE *cmd)
{
    FTP_COMMAND i;

	for ( i = 0; i < (FTP_COMMAND)FTP_COMMAND_TABLE_SIZE; i++ )
	{
		if ( !memcmppgm2ram((void*)cmd, (ROM void*)FTPCommandString[i], 3) )
			return i;
	}

	return FTP_CMD_UNKNOWN;
}

static void ParseFTPString(void)
{
    BYTE *p;
    BYTE v;
    enum { SM_FTP_PARSE_PARAM, SM_FTP_PARSE_SPACE } smParseFTP;

    smParseFTP  = SM_FTP_PARSE_PARAM;
    p           = (BYTE*)&FTPString[0];

    // Skip white blanks
    while( *p == ' ' )
        p++;

    FTP_argv[0]  = (BYTE*)p;
    FTP_argc     = 1;

    while( (v = *p) )
    {
        switch(smParseFTP)
        {
        case SM_FTP_PARSE_PARAM:
            if ( v == ' ' || v == ',' )
            {
                *p = '\0';
                smParseFTP = SM_FTP_PARSE_SPACE;
            }
            else if ( v == '\r' || v == '\n' )
                *p = '\0';
            break;

        case SM_FTP_PARSE_SPACE:
            if ( v != ' ' )
            {
                FTP_argv[FTP_argc++] = (BYTE*)p;
                smParseFTP = SM_FTP_PARSE_PARAM;
            }
            break;
        }
        p++;
        if(FTP_argc == MAX_FTP_ARGS)
        	break;
    }
}

#endif	// #if defined(STACK_USE_FTP_SERVER)
